# MCP4725 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2017-05-10 | [dnc40085](https://github.com/dnc40085) | [dnc40085](https://github.com/dnc40085) | [mcp4725.c](../../app/modules/mcp4725.c)|


This module provides access to the [MCP4725 12-bit Digital to Analog Converter](http://ww1.microchip.com/downloads/en/DeviceDoc/22039d.pdf).

!!!important:
	VDD is the power supply pin for the device. The voltage at the VDD pin is used as the supply input as well as the DAC reference input. The power supply at the VDD pin should be clean as possible for good DAC performance.

!!!note:
	The MCP4725 device address contains four fixed bits ( 1100 = device code) and three address bits (A2, A1, A0). The A2 and A1 bits are hard-wired during manufacturing, and A0 bit is determined by the logic state of A0 pin. The A0 pin can be connected to VDD or VSS , or actively driven by digital logic levels. The address pin(A0) can be actively driven by a GPIO to act as a chip select, allowing more than 2 devices to be used on the same bus.

## mcp4725.read()
Gets contents of the dac register and EEPROM.

#### Syntax
`mcp4725.read({[a0], [a1], [a2]})`

#### Parameters
- `A0` Address bit 0. This bit is user configurable via MCP4725 pin 6(A0). (valid states: 0 or 1) (default: 0)
- `A1` Address bit 1. This bit is hard-wired during manufacture. (valid states: 0 or 1) (default: 0)
	- Note: Modules purchased from Adafruit have this bit(A1) set high(1).
- `A2` Address bit 2. This bit is hard-wired during manufacture. (valid states: 0 or 1) (default: 0)

#### Returns
* `cur_pwrdn` Current power down configuration value.
* `cur_val` Current value stored in dac register.
* `eeprom_pwrdn` Power down configuration stored in EEPROM.
* `eeprom_val` DAC value stored in EEPROM.
* `eeprom_state` EEPROM write status
	* `0` EEPROM write is incomplete.
	* `1` EEPROM write has completed
* `por_state` Power-On-Reset status;
	* `0` The MCP4725 is performing reset and is not ready.
	* `1` The MCP4725 has successfully performed reset.

#### Example
```lua
-- Get current configuration using default i2c address 0x60(A0=0, A1=0, A2=0).
do
local ID = 0
local SDA = 6
local SCL = 5

i2c.setup(ID, SDA, SCL, i2c.SLOW)

local cur_pwrdn, cur_val, eeprom_pwrdn, eeprom_val, eeprom_state, por_state = mcp4725.read()

print("\n Current configuration:\n\tpower down value: "..cur_pwrdn.."\n\tdac value: "..cur_val)
print(" Configuration stored in EEPROM:\n\tpower down value: "..eeprom_pwrdn.."\n\tdac value: "..eeprom_val)
print(" EEPROM write state: "..(eeprom_state==1 and "Completed" or "incomplete"))
print(" Power-On-Reset state: "..(por_state==1 and "Completed" or "incomplete"))
end

-- Get current configuration using default i2c address 0x60(A0=0, A1=0, A2=0).
-- The MCP4725's address pin(A0) is being driven with gpio 4(pin 2).
do
local ID = 0
local SDA = 6
local SCL = 5
local mcp4725_chip_sel = 2

i2c.setup(ID, SDA, SCL, i2c.SLOW)
gpio.mode(mcp4725_chip_sel, gpio.OUTPUT, gpio.PULLUP)

gpio.write(mcp4725_chip_sel, 1)
local cur_pwrdn, cur_val, eeprom_pwrdn, eeprom_val, eeprom_state, por_state = mcp4725.read({A0=1})
gpio.write(mcp4725_chip_sel, 0)

print("\n Current configuration:\n\tpower down value: "..cur_pwrdn.."\n\tdac value: "..cur_val)
print(" Configuration stored in EEPROM:\n\tpower down value: "..eeprom_pwrdn.."\n\tdac value: "..eeprom_val)
print(" EEPROM write state: "..(eeprom_state==1 and "Completed" or "incomplete"))
print(" Power-On-Reset state: "..(por_state==1 and "Completed" or "incomplete"))
end
```
#### See also
- [`i2c.setup()`](i2c.md#i2csetup)


## mcp4725.write()
Write configuration to dac register or dac register and eeprom.

#### Syntax
`mcp4725.write({[a0], [a1], [a2], value, [pwrdn], [save]})`

#### Parameters
- `A0` Address bit 0. This bit is user configurable via MCP4725 pin 6(A0). (valid states: 0 or 1) (default: 0)
- `A1` Address bit 1. This bit is hard-wired during manufacture. (valid states: 0 or 1) (default: 0)
	- Note: Modules purchased from Adafruit have this bit(A1) set high(1).
- `A2` Address bit 2. This bit is hard-wired during manufacture. (valid states: 0 or 1) (default: 0)
- `value` The value to be used to configure DAC (and EEPROM). (Range: 0 - 4095)
- `pwrdn` Set power down bits.
	- `mcp4725.PWRDN_NONE` MCP4725 output enabled. (Default)
	- `mcp4725.PWRDN_1K` MCP4725 output disabled, output pulled to ground via 1K restistor.
	- `mcp4725.PWRDN_100K` MCP4725 output disabled, output pulled to ground via 100K restistor.
	- `mcp4725.PWRDN_500K` MCP4725 output disabled, output pulled to ground via 500K restistor.
- `save` Save pwrdn and dac values to EEPROM. (Values are loaded on power-up or during reset.)
	- `true` Save configuration to EEPROM.
	- `false` Do not save configuration to EEPROM. (Default)

#### Returns
nil

#### Example
```lua

-- Set current configuration using default i2c address 0x60(A0=0, A1=0, A2=0).
do
  local ID = 0
  local SDA = 6
  local SCL = 5

  i2c.setup(ID, SDA, SCL, i2c.SLOW)
  mcp4725.write({value=2048})
end

-- Set current configuration and save to EEPROM using default i2c address 0x60(A0=0, A1=0, A2=0).
do
  local ID = 0
  local SDA = 6
  local SCL = 5

  i2c.setup(ID, SDA, SCL, i2c.SLOW)
  mcp4725.write({value=2048, save=true})
end

-- Set current configuration using default i2c address 0x60(A0=0, A1=0, A2=0).
-- The MCP4725's address pin(A0) is being driven with gpio 4(pin 2).
do
  local ID = 0
  local SDA = 6
  local SCL = 5
  local mcp4725_chip_sel = 2

  i2c.setup(ID, SDA, SCL, i2c.SLOW)
  gpio.mode(mcp4725_chip_sel, gpio.OUTPUT, gpio.PULLUP)

  gpio.write(mcp4725_chip_sel, 1)
  mcp4725.read({A0=1, value})
  gpio.write(mcp4725_chip_sel, 0)
end
```
#### See also
- [`i2c.setup()`](i2c.md#i2csetup)
