# Si7021 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2017-04-19 | [fetchbot](https://github.com/fetchbot) | [fetchbot](https://github.com/fetchbot) | [si7021.c](../../app/modules/si7021.c)|

This module provides access to the Si7021 humidity and temperature sensor.

## si7021.firmware()
Read the internal firmware revision of the Si7021 sensor.

#### Syntax
`si7021.firmware()`

#### Parameters
none

#### Returns
`fwrev` Firmware version
* `0xFF` Firmware version 1.0
* `0x20` Firmware version 2.0

#### Example
```lua
local sda, scl = 6, 5
i2c.setup(0, sda, scl, i2c.SLOW) -- call i2c.setup() only once
si7021.setup()

fwrev = si7021.firmware()
print(string.format("FW: %X\r\n", fwrev))
```

## si7021.read()

#### Syntax
`si7021.read()`

#### Parameters
none

#### Returns
- `hum` humidity (see note below)
- `temp` temperature (see note below)
- `hum_dec` humidity decimal
- `temp_dec` temperature decimal

!!! note

	If using float firmware then `hum` and `temp` are floating point numbers. On an integer firmware, the final values have to be concatenated from `hum` and `hum_dec` / `temp` and `temp_dec`.

#### Example
```lua
local sda, scl = 6, 5
i2c.setup(0, sda, scl, i2c.SLOW) -- call i2c.setup() only once
si7021.setup()

hum, temp, hum_dec, temp_dec = si7021.read()

-- Integer firmware using this example
print(string.format("Humidity:\t\t%d.%03d\nTemperature:\t%d.%03d\n", hum, hum_dec, temp, temp_dec))

-- Float firmware using this example
print("Humidity: "..hum.."\n".."Temperature: "..temp)
```

## si7021.serial()
Read the individualized 64-bit electronic serial number of the Si7021 sensor.

#### Syntax
`si7021.serial()`

#### Parameters
none

#### Returns
- `sna` 32-bit serial number part a
- `snb` 32-bit serial number part b, upper byte contains the device identification
	* `0x00` or `0xFF` engineering samples
	* `0x0D` `13` Si7013
	* `0x14` `20` Si7020
	* `0x15` `21` Si7021

#### Example
```lua
local sda, scl = 6, 5
i2c.setup(0, sda, scl, i2c.SLOW) -- call i2c.setup() only once
si7021.setup()

sna, snb = si7021.serial()
print(string.format("SN:\t\t%X%X\nDevice:\tSi70%d", sna, snb, bit.rshift(snb,24)))
```

## si7021.setting()
Settings for the sensors configuration register to adjust measurement resolution, on-chip heater and read the supply voltage status.

#### Syntax
`si7021.setting(RESOLUTION[, HEATER, HEATER_SETTING])`

#### Parameters
- `RESOLUTION`
	* `si7021.RH12_TEMP14` Relative Humidity 12 bit - Temperature 14 bit (default)
	* `si7021.RH08_TEMP12` Relative Humidity 8 bit - Temperature 12 bit
	* `si7021.RH10_TEMP13` Relative Humidity 10 bit - Temperature 13 bit
	* `si7021.RH11_TEMP11` Relative Humidity 11 bit - Temperature 11 bit
- `HEATER` optional
	* `si7021.HEATER_ENABLE` On-chip Heater Enable
	* `si7021.HEATER_DISABLE` On-chip Heater Disable (default)
- `HEATER_SETTING` optional
	* `0x00` - `0x0F` 3.09 mA - 94.20 mA

#### Returns
- `resolution`
	* `0` Relative Humidity 12 bit - Temperature 14 bit
	* `1` Relative Humidity 8 bit - Temperature 12 bit
	* `2` Relative Humidity 10 bit - Temperature 13 bit
	* `3` Relative Humidity 11 bit - Temperature 11 bit
- `vdds`
	* `0` VDD OK (1.9V - 3.6V)
	* `1` VDD LOW (1.8V - 1.9V)
- `heater`
	* `0` Disabled
	* `1` Enabled
- `heater_setting`
	* `0` - `15`

#### Example
```lua
local id, sda, scl = 0, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW) -- call i2c.setup() only once
si7021.setup()

res, vdds, heater, heater_set = si7021.setting(si7021.RH12_TEMP14)
res, vdds, heater, heater_set = si7021.setting(si7021.RH12_TEMP14, si7021.HEATER_ENABLE, 0x01)
```

## si7021.setup()
Initializes the device on fixed I²C device address (0x40).

#### Syntax
`si7021.setup()`

#### Parameters
none

#### Returns
`nil`

#### Example
```lua
local sda, scl = 6, 5
i2c.setup(0, sda, scl, i2c.SLOW) -- call i2c.setup() only once
si7021.setup()
```
