# Lua MCP23017 Module for NodeMCU / ESP8266

| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2020-04-10 | [Marcel P.](https://github.com/plomi-net) | [Marcel P.](https://github.com/plomi-net) | [mcp23017.lua](../../lua_modules/mcp23017/mcp23017.lua) |


This Lua module provides access to the MCP23017 I²C I/O Expander.

The [MCP23017](http://ww1.microchip.com/downloads/en/devicedoc/20001952c.pdf) is a port expander and provides 16 channels for inputs and outputs. 
Up to 8 devices (128 channels) are possible by the configurable address (A0 - A2).

Due to the 16 channels, 2 bytes are required for switching outputs or reading input signals. These are A and B. 
A single pin can be set or a whole byte. 

The numbering of the individual pins starts at 0 and ends with 7.
The numbers are for each register GPIO A and GPIO B.


!!! important
	The module requires `i2c` and `bit` C module built into firmware.


### Require
```lua
mcp = require "mcp23017"
```

## Example Script
The example script can be found [here](../../lua_examples/mcp23017/mcp23017_example.lua)

## setup()
Configures the address of the module and tests the connection to the i2c bus. 
The i2c id is required for an existing i2c interface, alternatively the sda and scl pins can be specified.
Then this function will establish the connection.
Automatically resets the device state (see `mcp23017:reset()`)

#### Syntax
`mcp23017:setup(address, i2c_id)`

#### Parameter
- `address` address for MCP23017, default: 0x20 (should be between 0x20 and 0x27)
- `i2c_id`  id for the i2c bus connection (remember to call i2c.setup before)

#### Return
`true` if device found, otherwise `false`.

#### possible Errors
- `MCP23017 device on address not found`
- `MCP23017 address is out of range`

#### Example
```lua
local mcp23017 = require "mcp23017"

local address = 0x20
local cSCL = 1
local cSDA = 2
local i2c_instance = 0

-- setup i2c bus and create instance for mcp23017 (assigned to mcp)
i2c.setup(i2c_instance, cSDA, cSCL, i2c.SLOW)
local mcp = mcp23017(address, i2c_instance)
```

## setMode()
Set the mode of a single channel. This can be OUTPUT or INPUT.

#### Syntax
`mcp23017:setMode(register, pin, mode)`

#### Parameter
- `register` the side of channels (GPA or GPB)
- `pin` the number to be set for the channel (0-15)
- `mode` the mode for the channel. This can be `mcp23017.INPUT` or `mcp23017.OUTPUT`

#### Return
`true`, in case of error `nil`.

#### Example
```lua
-- set pin 7 and 8 to output (GPA7 and GPB0) and GPB1 to input 
mcp:setMode(mcp.GPA, 7, mcp.OUTPUT)
mcp:setMode(mcp.GPB, 0, mcp.OUTPUT)
mcp:setMode(mcp.GPB, 1, mcp.INPUT)
```

## setPin()
Set the state of a single channel. This can be HIGH or LOW.

#### Syntax
`mcp23017:setMode(register, pin, state)`

#### Parameter
- `register` the side of channels (GPA or GPB)
- `pin` the number to be set for the channel (0-15)
- `state` the state for the channel. This can be `mcp23017.HIGH` or `mcp23017.LOW`

#### Return
`true`, in case of error `nil`.

#### Example
```lua
-- set pin 7 to high (GPA7)
mcp:setPin(mcp.GPA, 7, mcp.HIGH)
-- set pin 8 to low (GPB0)
mcp:setPin(mcp.GPB, 0, mcp.LOW)
```

## getPinState()
get the state for a single channel. This can be HIGH or LOW.

#### Syntax
`mcp23017:getPinState(register, pin)`

#### Parameter
- `register` the side of channels (GPA or GPB)
- `pin` the number for which a state is to be queried (0-15)

#### Return
`true` for HIGH, `false` for LOW, in case of error `nil`.

#### Example
```lua
-- get the state for pin 9 (GPB1)
print(mcp:getPinState(mcp.GPB, 1))
```

## reset()
By calling this function, a safe state is established. 
All channels are set to input.
This function can be used for a panic program.

#### Syntax
`mcp23017:reset()`

#### Parameter
None

#### Return
None

#### Example
```lua
-- reset the mcp23017 to startup defaults
mcp:reset()
```

## setInternalPullUp()
Enable or disable the internal pullup resistors.

#### Syntax
`mcp23017:setInternalPullUp(register, byte)`

#### Parameter
- `register` the side of channels (GPA or GPB)
- `byte` byte to set the pullup resistors

#### Return
None

#### Example
```lua
-- enable all pullup resistors for GPA
print(mcp:setInternalPullUp(mcp.GPA, 0xFF))
-- disable all pullup resistors for GPA
print(mcp:setInternalPullUp(mcp.GPA, 0x00))
```

## writeIODIR()
Setup the mode of the channels with a whole byte.


#### Syntax
`mcp23017:writeIODIR(register, byte)`

#### Parameter
- `register` the side of channels (GPA or GPB)
- `byte` byte to set the mode for all channels for this register

#### Return
None

#### Example
```lua
-- set all GPA to input
print(mcp:writeIODIR(mcp.GPA, 0xFF))
-- set all GPA to output
print(mcp:writeIODIR(mcp.GPA, 0x00))
```

## writeGPIO()
Setup the output state of the channels with a whole byte.

#### Syntax
`mcp23017:writeGPIO(register, byte)`

#### Parameter
- `register` the side of channels (GPA or GPB)
- `byte` byte to set the state for all channels for this register

#### Return
None

#### Example
```lua
-- set all GPA to HIGH
print(mcp:writeGPIO(mcp.GPA, 0xFF))
-- set all GPA to LOW
print(mcp:writeGPIO(mcp.GPA, 0x00))
```

## readGPIO()
Read the input states of the channels with a whole byte.

#### Syntax
`mcp23017:readGPIO(register)`

#### Parameter
- `register` the side of channels (GPA or GPB)

#### Return
byte with states

#### Example
```lua
-- get states for GPA
print(mcp:readGPIO(mcp.GPA))
```

