# DS18B20 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2017-06-11 | [fetchbot](https://github.com/fetchbot) | [fetchbot](https://github.com/fetchbot) | [ds18b20.c](../../../app/modules/ds18b20.c)|

This module provides access to the DS18B20 1-Wire digital thermometer. Note that NodeMCU offers both a C module (this one) and [a Lua module for this sensor](https://github.com/nodemcu/nodemcu-firmware/tree/dev/lua_modules/ds18b20). See [#2003](https://github.com/nodemcu/nodemcu-firmware/pull/2003) for a discussion on the respective merits of them.

## ds18b20.read()
Issues a temperature conversion of all connected sensors on the onewire bus and returns the measurment results after a conversion delay in a callback function.
The returned measurements can be filtered through the ROM addresses passed as a table or by the family type.
The callback function gets invoked for every specified sensor.

#### Syntax
`ds18b20.read(CALLBACK, ROM[, FAMILY_ADDRESS])`

#### Parameters
- `CALLBACK` callback function executed for each sensor
	* e.g. `function(INDEX, ROM, RES, TEMP, TEMP_DEC, PAR) print(INDEX, ROM, RES, TEMP, TEMP_DEC, PAR) end`
- `ROM` table which contains the addresses for the specified sensors, or left empty to perform a onewire bus search for all sensors
	* e.g. `{"28:FF:FF:FF:FF:FF:FF:FF","28:FF:FF:FF:FF:FF:FF:FF"}`, `{}`
- `FAMILY_ADDRESS` optional to limit the search for devices to a specific family type
	* e.g `0x28`

#### Returns
`nil`

#### Callback function parameters
- `INDEX` index of the sensor on the bus
- `ROM` sensors 64-bit lasered rom code
	* `28:FF:FF:FF:FF:FF:FF:FF` LSB, 8-bit family code, 48-bit serial number, MSB 8-bit crc
- `RES` temperature resolution
- `TEMP` temperature
- `TEMP_DEC` temperature decimals for integer firmware
- `PAR` sensor parasitic flag

!!! note

	If using float firmware then `temp` is a floating point number. On an integer firmware, the final value has to be concatenated from `temp` and `temp_dec`.

#### Example
```lua
local ow_pin = 3
ds18b20.setup(ow_pin)

-- read all sensors and print all measurement results
ds18b20.read(
	function(ind,rom,res,temp,tdec,par)
		print(ind,string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")),res,temp,tdec,par)
	end,{});

-- read only sensors with family type 0x28 and print all measurement results
ds18b20.read(
	function(ind,rom,res,temp,tdec,par)
		print(ind,string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")),res,temp,tdec,par)
	end,{},0x28);

-- save device roms in a variable
local addr = {}
ds18b20.read(
	function(ind,rom,res,temp,tdec,par)
		addr[ind] = {string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)"))}
	end,{});

-- read only sensors listed in the variable addr
ds18b20.read(
	function(ind,rom,res,temp,tdec,par)
		print(ind,string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")),res,temp,tdec,par)
	end,addr);

-- print only parasitic sensors
ds18b20.read(
	function(ind,rom,res,temp,tdec,par)
		if (par == 1) then
			print(ind,string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")),res,temp,tdec,par)
		end
	end,{});

-- print if temperature is greater or less than a defined value
ds18b20.read(
	function(ind,rom,res,temp,tdec,par)
		if (t > 25) then
			print(ind,string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")),res,temp,tdec,par)
		end
		if (t < 20) then
			print(ind,string.format("%02X:%02X:%02X:%02X:%02X:%02X:%02X:%02X",string.match(rom,"(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+):(%d+)")),res,temp,tdec,par)
		end
	end,{});
```

## ds18b20.setting()
Configuration of the temperature resolution settings.

#### Syntax
`ds18b20.setting(ROM, RES)`

#### Parameters
- `ROM` table which contains the addresses for the specified sensors, or empty for all sensors
	* e.g. `{"28:FF:FF:FF:FF:FF:FF:FF","28:FF:FF:FF:FF:FF:FF:FF"}`, `{}`
- `RES` temperature bit resolution
	* `9` - `12`

#### Returns
`nil`

#### Example
```lua
local ow_pin = 3
ds18b20.setup(ow_pin)

ds18b20.setting({"28:FF:FF:FF:FF:FF:FF:FF","28:FF:FF:FF:FF:FF:FF:FF"}, 9)
```

## ds18b20.setup()
Initializes the onewire bus on the selected pin.

#### Syntax
`ds18b20.setup(OW_BUS_PIN)`

#### Parameters
- `OW_BUS_PIN`
	* `1` - `12`

#### Returns
`nil`

#### Example
```lua
local ow_pin = 3
ds18b20.setup(ow_pin)
```