# u8g2 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2017-06-02 | [Oli Kraus](https://github.com/olikraus/u8glib), [Arnim Läuger](https://github.com/devsaurus) | [Arnim Läuger](https://github.com/devsaurus) | [u8g2.c](../../components/modules/u8g2.c)|

U8g2 is a graphics library developed at [olikraus/u8g2](https://github.com/olikraus/u8g2) with support for many different displays. It is the successor of [U8glib](https://github.com/olikraus/u8glib) which is not developed any further. Please see [How to port U8g code](https://github.com/olikraus/u8g2/wiki/u8gvsu8g2) for generic porting instructions.

!!! note "BSD License for U8g2lib Code"
    Universal 8bit Graphics Library (http://code.google.com/p/u8g2/)

    Copyright (c) 2016, olikraus@gmail.com

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
    are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list 
      of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright notice, this 
      list of conditions and the following disclaimer in the documentation and/or other 
      materials provided with the distribution.


        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
        CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, 
        INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
        MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
        DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR 
        CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
        SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
        NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
        CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
        STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
        ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
        ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The NodeMCU firmware supports the following displays in I²C and SPI mode:

- ld7032 60x32
- sh1106 128x64
- sh1107 - variants 64x128, seeed 96x96, 128x128
- sh1108 160x160
- ssd0323 os128064
- ssd1305 128x32
- ssd1306 - variants 128x32, 128x64, 64x48, and 96x16
- ssd1309 128x64
- ssd1318 128x96, 128x96_xcp
- ssd1325 128x63
- ssd1326 er 256x32
- ssd1327 - variants 96x96, ea w128128, and midas 128x128
- st7567 64x32
- st7588 jlx12864
- st75256 - variants jlx256128, jlx256160, jlx240160, jlx25664, and jlx172104
- uc1601 128x32
- uc1604 jlx19264
- uc1608 - 240x180 and erc24064 variants
- ux1610 dogxl160
- uc1611 - variants dogm240, dogxl240, and ew50850

SPI only:

- hx1230 96x68
- il3820 v2 296x128
- ist3020 erc19264
- lc9081 - variants 160x80, 160x160, 240x128, and 240x64
- ls013b7dh03 128x128
- max7219 32x8
- nt7534 tg12864r
- pcd8544 84x48
- pcf8812 96x65
- sed1520 122x32
- ssd1322 nhd 256x64 and nhd 128x64 variants
- ssd1329 128x96
- ssd1606 172x72
- ssd1607 200x200, gd_200x200, ws_200x200
- st7565 - variants 64128n, dogm128/132, erc12864, erc12864_alt, lm6059, c12832/c12864, and zolen 128x64
- st7567 - variants 132x64, jlx12864, and enh_dg128064i
- st7586 - s028hn118a and erc240160 variants
- st75256 - jlx172104 and jlx256128 variants
- t6963 - variants 240x128, 240x64, 256x64, 128x64, and 160x80
- uc1701 - dogs102 and mini12864 variants

This integration uses full "RAM" memory buffer without picture loop and calls u8g2's `begin()` internally when creating a display object. It is based on [v2.25.10](https://github.com/olikraus/U8g2_Arduino/releases/tag/2.25.10).

## Overview

### Library Usage
The Lua bindings for this library closely follow u8g2's object oriented C++ API. Based on the u8g2 class, you create an object for your display type. The communication interface has to be initialized up front, refer to the examples below on how to do this.

SSD1306 via I²C:
```lua
sla = 0x3c
disp = u8g2.ssd1306_i2c_128x64_noname(id, sla)
```

SSD1306 via SPI:
```lua
cs  = 22
dc  = 16
res = 0 -- RES is optional YMMV
disp = u8g2.ssd1306_128x64_noname(bus, cs, dc, res)
```

This object provides all of u8g2's methods to control the display. Refer to [graphics_test.lua](../../lua_examples/u8g2/graphics_test.lua) to get an impression how this is achieved with Lua code. Visit the [u8g2 homepage](https://github.com/olikraus/u8g2) for technical details.

### Displays selection
I²C and HW SPI based displays with support in u8g2 can be enabled.

The procedure is different for ESP8266 and ESP32 platforms.

#### ESP8266
Add the desired entries to the I²C or SPI display tables in [app/include/u8g2_displays.h](../../app/include/u8g2_displays.h):
```c
#define U8G2_DISPLAY_TABLE_I2C \
  U8G2_DISPLAY_TABLE_ENTRY(u8g2_Setup_ssd1306_i2c_128x64_noname_f, ssd1306_i2c_128x64_noname) \

#define U8G2_DISPLAY_TABLE_SPI \
  U8G2_DISPLAY_TABLE_ENTRY(u8g2_Setup_ssd1306_128x64_noname_f, ssd1306_128x64_noname) \
  U8G2_DISPLAY_TABLE_ENTRY(u8g2_Setup_pcd8544_84x48_f, pcd8544_84x48) \
  U8G2_DISPLAY_TABLE_ENTRY(u8g2_Setup_pcf8812_96x65_f, pcf8812_96x65) \
```

Alternatively, you can define them as `U8G2_DISPLAY_TABLE_I2C_EXTRA` and `U8G2_DISPLAY_TABLE_SPI_EXTRA` in an external file to avoid changing the source tree. Include the extra file on the `make` command line:
```
make EXTRA_CCFLAGS='-include $(TOP_DIR)/my_extras.h'
```

#### ESP32
Enable the desired entries for I²C and SPI displays in u8g2's sub-menu (run `make menuconfig`).

### Fonts selection
U8g2 comes with a wide range of fonts for small displays. Fonts can be supplied as strings or compiled into the firmware image to decrease the RAM footprint. If compiled into the firmware they become available as `u8g2.<font_name>` in Lua.

The procedure is different for ESP8266 and ESP32 platforms.

#### ESP8266
Add the desired fonts to the font table in [app/include/u8g2_fonts.h](../../app/include/u8g2_fonts.h):
```c
#define U8G2_FONT_TABLE \
  U8G2_FONT_TABLE_ENTRY(font_6x10_tf) \
  U8G2_FONT_TABLE_ENTRY(font_unifont_t_symbols) \
```

Alternatively, you can define this as `U8G2_FONT_TABLE_EXTRA` in an external file to avoid changing the source tree. Include the extra file on the `make` command line:
```
make EXTRA_CCFLAGS='-include $(TOP_DIR)/my_extras.h'
```

#### ESP32
Add the desired fonts to the font selection sub-entry via `make menuconfig`.

### Bitmaps
XBM bitmaps are supplied as strings to `drawXBM()` in contrast to the source code based inclusion of XBMs in upstream u8g2 library. This off-loads all data handling from the u8g2 module to generic methods for binary files. See [graphics_test.lua](../../lua_examples/u8g2/graphics_test.lua).

Conversion of XBM bitmaps can be performed online with [Online-Utility's Image Converter](http://www.online-utility.org/image_converter.jsp): Convert from XBM to MONO format and upload the binary result.

## I²C Display Drivers
Initialize a display via I²C.

- `u8g2.ld7032_i2c_60x32()`
- `u8g2.sh1106_i2c_128x64_noname()`
- `u8g2.sh1106_i2c_128x64_vcomh0()`
- `u8g2.sh1107_i2c_64x128()`
- `u8g2.sh1107_i2c_seeed_96x96()`
- `u8g2.sh1107_i2c_128x128()`
- `u8g2.sh1108_i2c_160x160()`
- `u8g2.ssd0323_i2c_os128064()`
- `u8g2.ssd1305_i2c_128x32_noname()`
- `u8g2.ssd1306_i2c_128x32_univision()`
- `u8g2.ssd1306_i2c_128x64_noname()`
- `u8g2.ssd1306_i2c_128x64_vcomh0()`
- `u8g2.ssd1309_i2c_128x64_noname0()`
- `u8g2.ssd1309_i2c_128x64_noname2()`
- `u8g2.ssd1306_i2c_128x64_alt0()`
- `u8g2.ssd1306_i2c_64x48_er()`
- `u8g2.ssd1306_i2c_96x16_er()`
- `u8g2.ssd1318_i2c_128x96()`
- `u8g2.ssd1318_i2c_128x96_xcp()`
- `u8g2.ssd1325_i2c_nhd_128x64()`
- `u8g2.ssd1326_i2c_er_256x32()`
- `u8g2.ssd1327_i2c_seeed_96x96()`
- `u8g2.ssd1327_i2c_ea_w128128()`
- `u8g2.ssd1327_i2c_midas_128x128()`
- `u8g2.st7567_i2c_64x32()`
- `u8g2.st7588_i2c_jlx12864()`
- `u8g2.st75256_i2c_jlx25664()`
- `u8g2.st75256_i2c_jlx172104()`
- `u8g2.st75256_i2c_jlx240160()`
- `u8g2.st75256_i2c_jlx256128()`
- `u8g2.st75256_i2c_jlx256160()`
- `u8g2.uc1602_i2c_128X32()`
- `u8g2.uc1604_i2c_jlx19264()`
- `u8g2.uc1608_i2c_erc24064()`
- `u8g2.uc1608_i2c_240x128()`
- `u8g2.uc1610_i2c_ea_dogxl160()`
- `u8g2.uc1611_i2c_ea_dogm240()`
- `u8g2.uc1611_i2c_ea_dogxl240()`
- `u8g2.uc1611_i2c_ew50850()`

#### Syntax
`u8g2.ssd1306_i2c_128x64_noname(id, address[, cb_fn])`

#### Parameters
- `id` i2c interface id, see [i2c module](i2c.md)
- `address` I²C slave address of display (unshifted)
- `cb_fn` optional callback function, see [Framebuffer callback](#framebuffer-callback)

#### Returns
u8g2 display object

#### Example for ESP8266
```lua
id  = 0
sda = 5 -- GPIO14
scl = 6 -- GPIO12
sla = 0x3c
i2c.setup(id, sda, scl, i2c.SLOW)
disp = u8g2.ssd1306_i2c_128x64_noname(id, sla)
```

#### Example for ESP32
```lua
id  = i2c.HW0
sda = 16
scl = 17
sla = 0x3c
i2c.setup(id, sda, scl, i2c.FAST)
disp = u8g2.ssd1306_i2c_128x64_noname(id, sla)
```

## SPI Display Drivers
Initialize a display via Hardware SPI.

- `u8g2.hx1230_96x68()`
- `u8g2.il3820_v2_296x128()`
- `u8g2.ist3020_erc19264()`
- `u8g2.lc7981_160x80()`
- `u8g2.lc7981_160x80()`
- `u8g2.lc7981_160x80()`
- `u8g2.lc7981_160x80()`
- `u8g2.ld7032_60x32()`
- `u8g2.ls013b7dh03_128x128()`
- `u8g2.max7219_32x8()`
- `u8g2.nt7534_tg12864r()`
- `u8g2.pcd8544_84x48()`
- `u8g2.pcf8812_96x65()`
- `u8g2.sh1106_128x64_noname()`
- `u8g2.sh1106_128x64_vcomh0()`
- `u8g2.sh1107_64x128()`
- `u8g2.sh1107_seeed_96x96()`
- `u8g2.sh1107_128x128()`
- `u8g2.sh1108_160x160()`
- `u8g2.sh1122_256x64()`
- `u8g2.ssd0323_os128064()`
- `u8g2.ssd1305_128x32_noname()`
- `u8g2.ssd1306_128x32_univision()`
- `u8g2.ssd1306_128x64_noname()`
- `u8g2.ssd1306_128x64_vcomh0()`
- `u8g2.ssd1306_128x64_alt0()`
- `u8g2.ssd1306_64x48_er()`
- `u8g2.ssd1306_96x16_er()`
- `u8g2.ssd1309_128x64_noname0()`
- `u8g2.ssd1309_128x64_noname2()`
- `u8g2.ssd1318_128x96()`
- `u8g2.ssd1318_128x96_xcp()`
- `u8g2.ssd1322_nhd_128x64()`
- `u8g2.ssd1326_er_256x32()`
- `u8g2.ssd1327_ea_w128128()`
- `u8g2.ssd1327_midas_128x128()`
- `u8g2.ssd1322_nhd_256x64()`
- `u8g2.ssd1325_nhd_128x64()`
- `u8g2.ssd1327_seeed_96x96()`
- `u8g2.ssd1329_128x96_noname()`
- `u8g2.sed1520_122x32()`
- `u8g2.ssd1606_172x72()`
- `u8g2.ssd1607_200x200()`
- `u8g2.ssd1607_gd_200x200()`
- `u8g2.ssd1607_ws_200x200()`
- `u8g2.st7565_64128n()`
- `u8g2.st7565_ea_dogm128()`
- `u8g2.st7565_ea_dogm132()`
- `u8g2.st7565_erc12864()`
- `u8g2.st7565_erc12864_alt()`
- `u8g2.st7565_lm6059()`
- `u8g2.st7565_nhd_c12832()`
- `u8g2.st7565_nhd_c12864()`
- `u8g2.st7565_zolen_128x64()`
- `u8g2.st7567_enh_dg128064i()`
- `u8g2.st7567_64x32()`
- `u8g2.st7567_jxl12864()`
- `u8g2.st7567_pi_132x64()`
- `u8g2.st7586s_s028hn118a()`
- `u8g2.st7586s_erc240160()`
- `u8g2.st7588_jlx12864()`
- `u8g2.st7920_s_128x64()`
- `u8g2.st7920_s_192x32()`
- `u8g2.st75256_jlx25664()`
- `u8g2.st75256_jlx172104()`
- `u8g2.st75256_jlx240160()`
- `u8g2.st75256_jlx256128()`
- `u8g2.st75256_jlx256160()`
- `u8g2.t6963_240x128()`
- `u8g2.t6963_240x64()`
- `u8g2.t6963_256x64()`
- `u8g2.t6963_128x64()`
- `u8g2.t6963_160x80()`
- `u8g2.uc1601_128X32()`
- `u8g2.uc1604_jlx19264()`
- `u8g2.uc1608_240x128()`
- `u8g2.uc1608_erc24064()`
- `u8g2.uc1610_ea_dogxl160()`
- `u8g2.uc1611_ea_dogm240()`
- `u8g2.uc1611_ea_dogxl240()`
- `u8g2.uc1611_ew50850()`
- `u8g2.uc1701_ea_dogs102()`
- `u8g2.uc1701_mini12864()`

#### Syntax
`u8g2.ssd1306_128x64_noname(bus, cs, dc[, [res][, cb_fn]])`

#### Parameters
- `bus` SPI master bus
- `cs` GPIO pin for CS
- `dc` GPIO pin for DC
- `res` GPIO pin for RES, none if omitted
- `cb_fn` optional callback function, see [Framebuffer callback](#framebuffer-callback)

#### Returns
u8g2 display object

#### Example for ESP8266
```lua
-- Hardware SPI CLK  = GPIO14
-- Hardware SPI MOSI = GPIO13
-- Hardware SPI MISO = GPIO12 (not used)
-- Hardware SPI /CS  = GPIO15 (not used)
cs  = 8 -- GPIO15, pull-down 10k to GND
dc  = 4 -- GPIO2
res = 0 -- GPIO16
bus = 1
spi.setup(bus, spi.MASTER, spi.CPOL_LOW, spi.CPHA_LOW, 8, 8)
-- we won't be using the HSPI /CS line, so disable it again
gpio.mode(8, gpio.INPUT, gpio.PULLUP)
disp = u8g2.ssd1306_128x64_noname(bus, cs, dc, res)
```

#### Example for ESP32
```lua
sclk = 19
mosi = 23
cs   = 22
dc   = 16
res  = 17
bus = spi.master(spi.HSPI, {sclk=sclk, mosi=mosi})
disp = u8g2.ssd1306_128x64_noname(bus, cs, dc, res)
```

## Framebuffer callback
Each display type can be initialized to provide the framebuffer contents in run-length encoded format to a Lua callback. This mode is enabled when a callback function is specified for the setup function. Hardware display and framebuffer callback can be operated in parallel. If the callback function is the only parameter then no signals for a hardware display are generated, leaving a virtual display.

The callback function can be used to process the framebuffer line by line. It's called with either `nil` as parameter to indicate the start of a new frame or with a string containing a line of the framebuffer with run-length encoding. First byte in the string specifies how many pairs of (x, len) follow, while each pair defines the start (leftmost x-coordinate) and length of a sequence of lit pixels. All other pixels in the line are dark.

```lua
n = struct.unpack("B", rle_line)
print(n.." pairs")
for i = 0,n-1 do
  print(string.format("  x: %d len: %d", struct.unpack("BB", rle_line, 1+1 + i*2)))
end
```

#### Syntax
```lua
u8g2.ssd1306_i2c_128x64_noname(id, address[, cb_fn])
u8g2.ssd1306_i2c_128x64_noname(cb_fn)
u8g2.ssd1306_128x64_noname(bus, cs, dc[, [res][, cb_fn]])
u8g2.ssd1306_128x64_noname(cb_fn)
```

#### Parameters
- `id`, `address`, `bus`, `cs`, `dc`, `res` see above
- `cb_fn([rle_line])` callback function. `rle_line` is a string containing a run-length encoded framebuffer line, or `nil` to indicate start of frame.

## Constants
Constants for various functions.

`u8g2.DRAW_UPPER_RIGHT`, `u8g2.DRAW_UPPER_LEFT`, `u8g2.DRAW_LOWER_RIGHT`, `u8g2.DRAW_LOWER_LEFT`, `u8g2.DRAW_ALL`,

`u8g2.R0`, `u8g2.R1`, `u8g2.R2`, `u8g2.R3`, `u8g2.MIRROR`,

`u8g2.font_6x10`, ...


# u8g2.disp Sub-Module

## u8g2.disp:clearBuffer()
Clears all pixel in the memory frame buffer.

See [u8g2 clearBuffer()](https://github.com/olikraus/u8g2/wiki/u8g2reference#clearbuffer).

## u8g2.disp:drawBox()
Draw a box (filled frame).

See [u8g2 drawBox()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawbox).

## u8g2.disp:drawCircle()
Draw a circle.

See [u8g2 drawCircle()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawcircle).

Note that parameter `opt` is optional and defaults to `u8g2.DRAW_ALL` if omitted.

## u8g2.disp:drawDisc()
Draw a filled circle.

See [u8g2 drawDisc()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawdisc).

Note that parameter `opt` is optional and defaults to `u8g2.DRAW_ALL` if omitted.

## u8g2.disp:drawEllipse()
Draw an ellipse.

See [u8g2 drawEllipse()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawellipse).

Note that parameter `opt` is optional and defaults to `u8g2.DRAW_ALL` if omitted.

## u8g2.disp:drawFilledEllipse()
Draw a filled ellipse.

See [u8g2 drawFilledEllipse()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawfilledellipse).

Note that parameter `opt` is optional and defaults to `u8g2.DRAW_ALL` if omitted.

## u8g2.disp:drawFrame()
Draw a frame (empty box).

See [u8g2 drawFrame()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawframe).

## u8g2.disp:drawGlyph()
Draw a single character.

See [u8g2 drawGlyph()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawglyph).

## u8g2.disp:drawHLine()
Draw a horizontal line.

See [u8g2 drawHLine()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawhline).

## u8g2.disp:drawLine()
Draw a line between two points.

See [u8g2 drawLine()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawline).

## u8g2.disp:drawPixel()
Draw a pixel.

See [u8g2 drawPixel()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawpixel).

## u8g2.disp:drawRBox()
Draw a box with round edges.

See [u8g2 drawRBox()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawrbox).

## u8g2.disp:drawRFrame()
Draw a frame with round edges.

See [u8g2 drawRFrame()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawrframe).

## u8g2.disp:drawStr()
Draw a string.

See [u8g2 drawStr()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawstr).

## u8g2.disp:drawTriangle()
Draw a triangle (filled polygon).

See [u8g2 drawTriangle()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawtriangle).

## u8g2.disp:drawUTF8()
Draw a string which is encoded as UTF-8.

See [u8g2 drawUTF8()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawutf8).

## u8g2.disp:drawVLine()
Draw a vertical line.

See [u8g2 drawVLine()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawvline).

## u8g2.disp:drawXBM()
Draw a XBM Bitmap.

See [u8g2 drawXBM()](https://github.com/olikraus/u8g2/wiki/u8g2reference#drawxbm).

XBM bitmaps are supplied as strings to `drawXBM()`. This off-loads all data handling from the u8g2 module to generic methods for binary files. See [graphics_test.lua](../../lua_examples/u8glib/u8g_graphics_test.lua).

In contrast to the source code based inclusion of XBMs in upstream u8g2 library, it's required to provide precompiled binary files. This can be performed online with [Online-Utility's Image Converter](http://www.online-utility.org/image_converter.jsp): Convert from XBM to MONO format and upload the binary result.

## u8g2.disp:getAscent()
Returns the reference height of the glyphs above the baseline (ascent).

See [u8g2 getAscent()](https://github.com/olikraus/u8g2/wiki/u8g2reference#getascent).

## u8g2.disp:getDescent()
Returns the reference height of the glyphs below the baseline (descent).

See [u8g2 getDescent()](https://github.com/olikraus/u8g2/wiki/u8g2reference#getdescent).

## u8g2.disp:getStrWidth()
Return the pixel width of string.

See [u8g2 getStrWidth()](https://github.com/olikraus/u8g2/wiki/u8g2reference#getstrwidth).

## u8g2.disp:getUTF8Width()
Return the pixel width of an UTF-8 encoded string.

See [u8g2 getUTF8Width()](https://github.com/olikraus/u8g2/wiki/u8g2reference#getutf8width).

## u8g2.disp:sendBuffer()
Send the content of the memory frame buffer to the display.

See [u8g2 sendBuffer()](https://github.com/olikraus/u8g2/wiki/u8g2reference#sendbuffer).

## u8g2.disp:setBitmapMode()
Define bitmap background color mode.

See [u8g2 setBitmapMode()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setbitmapmode).

## u8g2.disp:setContrast()
Set the contrast or brightness.

See [u8g2 setContrast()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setconstrast).

## u8g2.disp:setDisplayRotation()
Changes the display rotation.

See [u8g2 setDisplayRotation()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setdisplayrotation).

## u8g2.disp:setDrawColor()
Defines the bit value (color index) for all drawing functions.

See [u8g2 setDrawColor()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setdrawcolor).

## u8g2.disp:setFlipMode()
Set flip (180 degree rotation) mode.

See [u8g2 setFlipMode()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setflipmode).

## u8g2.disp:setFont()
Define a u8g2 font for the glyph and string drawing functions. They can be supplied as strings or compiled into the firmware image. They are available as `u8g2.<font_name>` in Lua.

See [u8g2 setFont()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfont).

## u8g2.disp:setFontDirection()
Set the drawing direction of all strings or glyphs.

See [u8g2 setFontDirection()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontdirection).

## u8g2.disp:setFontMode()
Define font background color mode.

See [u8g2 setFontMode()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontmode).

## u8g2.disp:setFontPosBaseline()
Change the reference position for the glyph and string draw functions to "baseline".

See [u8g2 setFontPosBaseline()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontposbaseline).

## u8g2.disp:setFontPosBottom()
Change the reference position for the glyph and string draw functions to "bottom".

See [u8g2 setFontPosBottom()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontposbottom).

## u8g2.disp:setFontPosTop()
Change the reference position for the glyph and string draw functions to "top".

See [u8g2 setFontPosTop()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontpostop).

## u8g2.disp:setFontPosCenter()
Change the reference position for the glyph and string draw functions to "center".

See [u8g2 setFontPosCenter()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontposcenter).

## u8g2.disp:setFontRefHeightAll()
Set ascent and descent calculation mode to "highest and lowest glyph".

See [u8g2 setFontRefHeightAll()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontrefheightall).

## u8g2.disp:setFontRefHeightExtendedText()
Set ascent and descent calculation mode to "highest of [A1(], lowest of [g(]".

See [u8g2 setFontRefHeightExtendedText()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontrefheightextendedtext).

## u8g2.disp:setFontRefHeightText()
Set ascent and descent calculation mode to "highest of [A1], lowest of [g]".

See [u8g2 setFontRefHeightText()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setfontrefheighttext).

## u8g2.disp:setPowerSave()
Activate or disable power save mode of the display.

See [u8g2 setPowerSave()](https://github.com/olikraus/u8g2/wiki/u8g2reference#setpowersave).
