# ADS1115 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2017-04-24 | [fetchbot](https://github.com/fetchbot) | [fetchbot](https://github.com/fetchbot) | [ads1115.c](../../../app/modules/ads1115.c)|

This module provides access to the ADS1115 16-Bit analog-to-digital converter.

!!! caution

	The **ABSOLUTE MAXIMUM RATINGS** for all analog inputs are `–0.3V to VDD+0.3V` referred to GND.

## ads1115.read()
Gets the result stored in the register of a previously issued conversion, e.g. in continuous mode or with a conversion ready interrupt.

#### Syntax
`volt, volt_dec, adc = ads1115.read()`

#### Parameters
none

#### Returns
- `volt` voltage in mV (see note below)
- `volt_dec` voltage decimal (see note below)
- `adc` raw adc value

!!! note

	If using float firmware then `volt` is a floating point number. On an integer firmware, the final value has to be concatenated from `volt` and `volt_dec`.

#### Example
```lua
local id, alert_pin, sda, scl = 0, 7, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.setup(ads1115.ADDR_GND)

-- continuous mode
ads1115.setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.CONTINUOUS)
-- read adc result with read()
volt, volt_dec, adc = ads1115.read()
print(volt, volt_dec, adc)

-- comparator
ads1115.setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.CONTINUOUS, ads1115.COMP_1CONV, 1000, 2000)
local function comparator(level, when)
	-- read adc result with read() when threshold reached
	volt, volt_dec, adc = ads1115.read()
	print(volt, volt_dec, adc)
end
gpio.mode(alert_pin, gpio.INT)
gpio.trig(alert_pin, "both", comparator)
-- read adc result with read()
volt, volt_dec, adc = ads1115.read()
print(volt, volt_dec, adc)
```

## ads1115.setting()
Configuration settings for the ADC.

#### Syntax
`ads1115.setting(GAIN, SAMPLES, CHANNEL, MODE[, CONVERSION_RDY][, COMPARATOR, THRESHOLD_LOW, THRESHOLD_HI])`

#### Parameters
- `GAIN` Programmable gain amplifier
	* `ads1115.GAIN_6_144V` 2/3x Gain
	* `ads1115.GAIN_4_096V` 1x Gain
	* `ads1115.GAIN_2_048V` 2x Gain
	* `ads1115.GAIN_1_024V` 4x Gain
	* `ads1115.GAIN_0_512V` 8x Gain
	* `ads1115.GAIN_0_256V` 16x Gain
- `SAMPLES` Data rate in samples per second
	* `ads1115.DR_8SPS`
	* `ads1115.DR_16SPS`
	* `ads1115.DR_32SPS`
	* `ads1115.DR_64SPS`
	* `ads1115.DR_128SPS`
	* `ads1115.DR_250SPS`
	* `ads1115.DR_475SPS`
	* `ads1115.DR_860SPS`
- `CHANNEL` Input multiplexer for single-ended or differential measurement
	* `ads1115.SINGLE_0` channel 0 to GND
	* `ads1115.SINGLE_1` channel 1 to GND
	* `ads1115.SINGLE_2` channel 2 to GND
	* `ads1115.SINGLE_3` channel 3 to GND
	* `ads1115.DIFF_0_1` channel 0 to 1
	* `ads1115.DIFF_0_3` channel 0 to 3
	* `ads1115.DIFF_1_3` channel 1 to 3
	* `ads1115.DIFF_2_3` channel 2 to 3
- `MODE` Device operating mode
	* `ads1115.SINGLE_SHOT` single-shot mode
	* `ads1115.CONTINUOUS` continuous mode
- `CONVERSION_RDY` Number of conversions after conversion ready asserts (optional)
	* `ads1115.CONV_RDY_1`
	* `ads1115.CONV_RDY_2`
	* `ads1115.CONV_RDY_4`
- `COMPARATOR` Number of conversions after comparator asserts (optional)
	* `ads1115.COMP_1CONV`
	* `ads1115.COMP_2CONV`
	* `ads1115.COMP_4CONV`
- `THRESHOLD_LOW`
	* `0` - `+ GAIN_MAX` in mV for single-ended inputs
	* `- GAIN_MAX` - `+ GAIN_MAX` in mV for differential inputs
- `THRESHOLD_HI`
	* `0` - `+ GAIN_MAX` in mV for single-ended inputs
	* `- GAIN_MAX` - `+ GAIN_MAX` in mV for differential inputs

#### Returns
`nil`

#### Example
```lua
local id, sda, scl = 0, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.setup(ads1115.ADDR_GND)

ads1115.setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.SINGLE_SHOT)
```

## ads1115.setup()
Initializes the device on the defined I²C device address.

#### Syntax
`ads1115.setup(ADDRESS)`

#### Parameters
- `ADDRESS`
	* `ads1115.ADDR_GND`
	* `ads1115.ADDR_VDD`
	* `ads1115.ADDR_SDA`
	* `ads1115.ADDR_SCL`

#### Returns
`nil`

#### Example
```lua
local id, sda, scl = 0, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)

ads1115.setup(ads1115.ADDR_GND)
```

## ads1115.startread()
Starts the ADC reading for single-shot mode and after the conversion is done it will invoke an optional callback function in which the ADC conversion result can be obtained.

#### Syntax
`ads1115.startread([CALLBACK])`

#### Parameters
- `CALLBACK` callback function which will be invoked after the adc conversion is done
	* `function(volt, volt_dec, adc) end`

#### Returns
- `nil`

#### Example
```lua
local id, alert_pin, sda, scl = 0, 7, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.setup(ads1115.ADDR_GND)

-- single shot
ads1115.setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.SINGLE_SHOT)
-- start adc conversion and get result in callback after conversion is ready
ads1115.startread(function(volt, volt_dec, adc) print(volt, volt_dec, adc) end)

-- conversion ready
ads1115.setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.SINGLE_SHOT, ads1115.CONV_RDY_1)
local function conversion_ready(level, when)
	volt, volt_dec, adc = ads1115.read()
	print(volt, volt_dec, adc)
end
gpio.mode(alert_pin, gpio.INT)
gpio.trig(alert_pin, "down", conversion_ready)
-- start conversion and get result with read() after conversion ready pin asserts
ads1115.startread()
```
