# HX711 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2015-10-09 | [Chris Takahashi](https://github.com/christakahashi) | [Chris Takahashi](https://github.com/christakahashi) | [hx711.c](../../app/modules/hx711.c)|
| 2019-04-20 | [Philip Gladstone](https://github.com/pjsg) | [Philip Gladstone](https://github.com/pjsg) 

This module provides access to an [HX711 load cell amplifier/ADC](https://learn.sparkfun.com/tutorials/load-cell-amplifier-hx711-breakout-hookup-guide). The HX711 is an inexpensive 24bit ADC with programmable 128x, 64x, and 32x gain. The standard Chinese sources have [cheap HX711 boards](https://www.aliexpress.com/wholesale?SearchText=hx711+module) for around $1.

This can be used for single shot reads, or repetitive reads.

Note: To save ROM image space, this module is not compiled into the firmware by default.

## hx711.init()

Initialize io pins for hx711 clock and data.

#### Syntax
`hx711.init(clk, data)`

#### Parameters
- `clk` pin the hx711 clock signal is connected to
- `data` pin the hx711 data signal is connected to

#### Returns
`nil`

#### Example
```lua
-- Initialize the hx711 with clk on pin 5 and data on pin 6
hx711.init(5,6)
```

## hx711.read()

Read digital loadcell ADC value.

#### Syntax
`hx711.read(mode)`

#### Parameters
- `mode` ADC mode.  This parameter specifies which input and the gain to apply to that input. Reading in mode 1 or 2 takes longer than reading in mode 0.

|mode | channel | gain |
|-----|---------|------|
| 0   | A       | 128  |
| 1   | B       | 32  |
| 2   | A       | 64  |

#### Returns
a number (24 bit signed ADC value extended to the machine int size)

#### Example
```lua
-- Read ch A with 128 gain.
raw_data = hx711.read(0)
```

## hx711.start()

Starts to read multiple samples from the ADC. 

#### Syntax
`hx711.start(mode, samples, callback)`

#### Parameters
- `mode` ADC mode.  This parameter is currently ignored and reserved to ensure backward compatibility if support for additional modes is added. 
- `samples` The number of samples before the callback is invoked. The length of time depends on the chip's sampling rate.
- `callback` The callback is invoked with three arguments (see below).

|mode | channel | gain |
|-----|---------|------|
| 0   | A       | 128  |
| 1   | B       | 32  |
| 2   | A       | 64  |

#### Returns
nothing

#### Callback
This is invoked every time `samples` samples are read from the HX711. The arguments are:

- A string which contains `samples` packed 24 bit values. This can be unpacked with the `struct` module (using the "i3" format).
- The time in microseconds of the reception of the last sample in the buffer.
- The number of samples dropped before the start of this buffer (after the end of the previous buffer).

#### Notes
This api only is built if GPIO_INTERRUPT_ENABLE and GPIO_INTERRUPT_HOOK_ENABLE are defined in the
`user_config.h`. This is the default.

Also, do not try and mix calls to `start` and calls to `read`. Any calls to `read` will implicitly call `stop` first.

#### Example
```lua
-- Read ch A with 128 gain.
hx711.start(0, 2, function(s, t, d) local r1, r2, _ = struct.unpack("i3 i3", s) print(r1, r2) end)
```

## hx711.stop()

Stops a previously started set of reads. Any data in buffers is lost. No more callbacks will be invoked.

#### Syntax
`hx711.stop()`

#### Returns
nothing

