# ADS1115 Module
| Since  | Origin / Contributor  | Maintainer  | Source  |
| :----- | :-------------------- | :---------- | :------ |
| 2017-04-24 | [fetchbot](https://github.com/fetchbot) | [fetchbot](https://github.com/fetchbot) | [ads1115.c](../../app/modules/ads1115.c)|

This module provides access to the ADS1115 (16-Bit) and ADS1015 (12-Bit) analog-to-digital converters.
Other chips from the same family (ADS1113, ADS1114, ADS1013 and ADS1014) are likely to work. Missing hardware features will be silently ignored.

This module supports multiple devices connected to I²C bus. The devices of different types can be mixed.
The addressing of ADS family allows for maximum of 4 devices connected to the same I²C bus.

!!! caution

	The **ABSOLUTE MAXIMUM RATINGS** for all analog inputs are `–0.3V to VDD+0.3V` referred to GND.



## ads1115.ads1115()
Registers ADS1115 (ADS1113, ADS1114) device.

#### Syntax
`ads1115.ADS1115(I2C_ID, I2C_ADDR)`

#### Parameters
- `I2C_ID` - always 0
- `ADDRESS` - I²C address of a device
	* `ads1115.ADDR_GND`
	* `ads1115.ADDR_VDD`
	* `ads1115.ADDR_SDA`
	* `ads1115.ADDR_SCL`

#### Returns
Registered `device` object

#### Example
```lua
local id, sda, scl = 0, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.reset()
adc1 = ads1115.ads1115(id, ads1115.ADDR_GND)
```


## ads1115.ads1015()
Registers ADS1015 (ADS1013, ADS1014) device.

#### Syntax
`ads1115.ads1015(I2C_ID, I2C_ADDR)`

#### Parameters
- `I2C_ID` - always 0
- `ADDRESS` - I²C address of a device
	* `ads1115.ADDR_GND`
	* `ads1115.ADDR_VDD`
	* `ads1115.ADDR_SDA`
	* `ads1115.ADDR_SCL`

#### Returns
Registered `device` object

#### Example
```lua
local id, sda, scl = 0, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.reset()
adc1 = ads1115.ads1015(id, ads1115.ADDR_VDD)
adc2 = ads1115.ads1115(id, ads1115.ADDR_SDA)
```
## ads1115.reset()
Reset all devices connected to I²C interface.

### Syntax
ads1115.reset()

#### Parameters
none

#### Returns
`nil`

#### Example
```lua
local id, alert_pin, sda, scl = 0, 7, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.reset()
```


# ADS Device

## ads1115.device:read()
Gets the result stored in the register of a previously issued conversion, e.g. in continuous mode or with a conversion ready interrupt.

#### Syntax
`volt, volt_dec, raw, sign = device:read()`

#### Parameters
none

#### Returns
- `volt` voltage in mV (see note below)
- `volt_dec` voltage decimal in uV (see note below)
- `adc` raw adc register value
- `sign` sign of the result (see note below)

!!! note

	If using float firmware then `volt` is a floating point number, `volt_dec` and `sign` are nil. On an integer firmware, the final value has to be concatenated from `volt`, `volt_dec` and `sign`. On integer firmware `volt` and `volt_dec` are always positive, sign can be `-1`, `0`, `1`.

#### Example
```lua
local id, alert_pin, sda, scl = 0, 7, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.reset()
adc1 = ads1115.ads1115(id, ads1115.ADDR_GND)

-- continuous mode
adc1:setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.CONTINUOUS)
-- read adc result with read()
volt, volt_dec, adc, sign = ads1:read()
print(volt, volt_dec, adc, sign)

-- comparator
adc1:setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.CONTINUOUS, ads1115.COMP_1CONV, 1000, 2000)
local function comparator(level, when)
	-- read adc result with read() when threshold reached
	gpio.trig(alert_pin)
	volt, volt_dec, adc, sign = ads1:read()
	print(volt, volt_dec, adc, sign)
end
gpio.mode(alert_pin, gpio.INT)
gpio.trig(alert_pin, "both", comparator)

-- read adc result with read()
volt, volt_dec, adc, sign = ads1115:read()
print(volt, volt_dec, adc, sing)

-- format value in int build
if sign then
    -- int build
    print(string.format("%s%d.%03d mV", sign >= 0 and "+" or "-", volt, volt_dec))
else
    -- float build
    -- just use V as it is
end
```


## ads1115.device:setting()
Configuration settings for the ADC.

#### Syntax
`device:setting(GAIN, SAMPLES, CHANNEL, MODE[, CONVERSION_RDY][, COMPARATOR, THRESHOLD_LOW, THRESHOLD_HI[,COMP_MODE]])`

#### Parameters
- `GAIN` Programmable gain amplifier
	* `ads1115.GAIN_6_144V` 2/3x Gain
	* `ads1115.GAIN_4_096V` 1x Gain
	* `ads1115.GAIN_2_048V` 2x Gain
	* `ads1115.GAIN_1_024V` 4x Gain
	* `ads1115.GAIN_0_512V` 8x Gain
	* `ads1115.GAIN_0_256V` 16x Gain
- `SAMPLES` Data rate in samples per second
	* `ads1115.DR_8SPS` ADS1115 only
	* `ads1115.DR_16SPS` ADS1115 only
	* `ads1115.DR_32SPS` ADS1115 only
	* `ads1115.DR_64SPS` ADS1115 only
	* `ads1115.DR_128SPS`
	* `ads1115.DR_250SPS`
	* `ads1115.DR_475SPS` ADS1115 only
	* `ads1115.DR_490SPS` ADS1015 only
	* `ads1115.DR_860SPS` ADS1115 only
	* `ads1115.DR_920SPS` ADS1015 only
	* `ads1115.DR_1600SPS` ADS1015 only
	* `ads1115.DR_2400SPS` ADS1015 only
	* `ads1115.DR_3300SPS` ADS1015 only
- `CHANNEL` Input multiplexer for single-ended or differential measurement
	* `ads1115.SINGLE_0` channel 0 to GND
	* `ads1115.SINGLE_1` channel 1 to GND
	* `ads1115.SINGLE_2` channel 2 to GND
	* `ads1115.SINGLE_3` channel 3 to GND
	* `ads1115.DIFF_0_1` channel 0 to 1
	* `ads1115.DIFF_0_3` channel 0 to 3
	* `ads1115.DIFF_1_3` channel 1 to 3
	* `ads1115.DIFF_2_3` channel 2 to 3
- `MODE` Device operating mode
	* `ads1115.SINGLE_SHOT` single-shot mode
	* `ads1115.CONTINUOUS` continuous mode
- `CONVERSION_RDY` Number of conversions after conversion ready asserts (optional)
	* `ads1115.CONV_RDY_1`
	* `ads1115.CONV_RDY_2`
	* `ads1115.CONV_RDY_4`
- `COMPARATOR` Number of conversions after comparator asserts (optional)
	* `ads1115.COMP_1CONV`
	* `ads1115.COMP_2CONV`
	* `ads1115.COMP_4CONV`
- `THRESHOLD_LOW`
	* `0` - `+ GAIN_MAX` in mV for single-ended inputs
	* `- GAIN_MAX` - `+ GAIN_MAX` in mV for differential inputs
- `THRESHOLD_HI`
	* `0` - `+ GAIN_MAX` in mV for single-ended inputs
	* `- GAIN_MAX` - `+ GAIN_MAX` in mV for differential inputs
- `COMP_MODE` Comparator mode
	* `ads1115.CMODE_TRAD` traditional comparator mode (with hysteresis)
	* `ads1115.CMODE_WINDOW` window comparator mode

note: Comparator and conversion ready are always configured to non-latching, active low.

#### Returns
`nil`

#### Example
```lua
local id, sda, scl = 0, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.reset()
adc1 = ads1115.ads1015(id, ads1115.ADDR_GND)

adc1:setting(ads1115.GAIN_6_144V, ads1115.DR_3300SPS, ads1115.SINGLE_0, ads1115.SINGLE_SHOT)
```


## ads1115.device:startread()
Starts the ADC reading for single-shot mode and after the conversion is done it will invoke an optional callback function in which the ADC conversion result can be obtained.

#### Syntax
`device:startread([CALLBACK])`

#### Parameters
- `CALLBACK` callback function which will be invoked after the adc conversion is done
	* `function(volt, volt_dec, adc, sign) end`

#### Returns
- `nil`

#### Example
```lua
local id, alert_pin, sda, scl = 0, 7, 6, 5
i2c.setup(id, sda, scl, i2c.SLOW)
ads1115.reset()
adc1 = ads1115.ads1115(id, ads1115.ADDR_VDD)

-- single shot
adc1:setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.SINGLE_SHOT)
-- start adc conversion and get result in callback after conversion is ready
adc1:startread(function(volt, volt_dec, adc, sign) print(volt, volt_dec, adc, sign) end)

-- conversion ready
adc1:setting(ads1115.GAIN_6_144V, ads1115.DR_128SPS, ads1115.SINGLE_0, ads1115.SINGLE_SHOT, ads1115.CONV_RDY_1)
local function conversion_ready(level, when)
	gpio.trig(alert_pin)
	volt, volt_dec, adc, sign = adc1:read()
	print(volt, volt_dec, adc, sign)
end
gpio.mode(alert_pin, gpio.INT)
gpio.trig(alert_pin, "down", conversion_ready)
-- start conversion and get result with read() after conversion ready pin asserts
adc1:startread()
```
